<?php
/*
Plugin Name: WooCommerce MagicPay Payment Gateway
Description: A plugin to add a custom payment option named MagicPay for WooCommerce. 
Version: 1.99
Author: Card2Krypto
*/
include_once plugin_dir_path(__FILE__) . 'shortcodes.php';


define('MAGICPAY_PLUGIN_VERSION', '1.99');
define('MAGICPAY_API_TOKEN', 'dGhpc19pc19hX3NhZmVfYXBpX2tleV9leGFtcGxlXzEyMzQ=');


if (!defined('ABSPATH')) {
    exit;
}

add_action('wp_enqueue_scripts', 'magicpay_enqueue_styles');
function magicpay_enqueue_styles()
{
    if (is_checkout()) {
        wp_enqueue_style(
            'magicpay-styles',
            plugins_url('assets/style.css', __FILE__),
            array(),
            '1.0'
        );
    }
}


add_action('admin_menu', 'magicpay_add_admin_menu');
function magicpay_add_admin_menu()
{
    add_menu_page(
        'MagicPay Settings',
        'MagicPay',
        'manage_options',
        'magicpay-settings',
        'magicpay_settings_page'
    );
}

function magicpay_settings_page()
{
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $order_prefix = sanitize_text_field($_POST['magicpay_order_prefix']);
        if (!preg_match('/^[a-zA-Z]{1,3}$/', $order_prefix)) {
            echo '<div class="error"><p>Order ID Prefix must be 1 to 3 letters only (A-Z or a-z).</p></div>';
        } else {
            update_option('magicpay_order_prefix', $order_prefix);
            update_option('magicpay_shopid', sanitize_text_field($_POST['magicpay_shopid']));
            update_option('magicpay_shopname', sanitize_text_field($_POST['magicpay_shopname']));
            update_option('magicpay_payment_url', esc_url_raw($_POST['magicpay_payment_url']));
            update_option('magicpay_auto_reminder', isset($_POST['magicpay_auto_reminder']) ? 'yes' : 'no');
            echo '<div class="updated"><p>MagicPay settings updated successfully!</p></div>';
        }
    }
    $shopid = get_option('magicpay_shopid', '');
    $payment_url = "https://magicpayment.online/dashboard/quikipay.php"; 
    $auto_reminder = get_option('magicpay_auto_reminder', 'no');
    $shopname = get_option('magicpay_shopname', '');
    $order_prefix = get_option('magicpay_order_prefix', '');
    if (empty($shopname)) {
        $shopname = parse_url(home_url(), PHP_URL_HOST); 
    }
?>
    <div class="wrap">
        <h1>MagicPay Settings</h1>
        <form method="POST">
            <label for="magicpay_shopid">Shop ID:</label>
            <input type="text" id="magicpay_shopid" name="magicpay_shopid" value="<?php echo esc_attr($shopid); ?>" required><br><br>

            <label for="magicpay_shopname">Shop Name:</label>
            <input type="text" id="magicpay_shopname" name="magicpay_shopname" value="<?php echo esc_attr($shopname); ?>" required><br><br>

            <label for="magicpay_order_prefix">Order ID Prefix (Required):</label>
            <input type="text" id="magicpay_order_prefix" name="magicpay_order_prefix" value="<?php echo esc_attr($order_prefix); ?>" required><br><br>

            <label for="magicpay_payment_url">Dashboard Tracker URL:</label>
            <input type="text" id="magicpay_payment_url" name="magicpay_payment_url" value="<?php echo esc_url($payment_url); ?>" required readonly><br><br>

            <label for="magicpay_auto_reminder">
                <input type="checkbox" id="magicpay_auto_reminder" name="magicpay_auto_reminder" <?php checked($auto_reminder, 'yes'); ?>>
                Auto-pay reminder after 15 min
            </label><br><br>

            <p><strong>Plugin Version:</strong> <?php echo esc_html(MAGICPAY_PLUGIN_VERSION); ?></p>

            <input type="submit" value="Save Settings" class="button button-primary">
        </form>
    </div>
    <?php
}

add_filter('woocommerce_order_number', 'magicpay_modify_order_number');
function magicpay_modify_order_number($order_id)
{
    $prefix = get_option('magicpay_order_prefix', '');
    if (!empty($prefix)) {
        return $prefix . $order_id;
    }
    return $order_id;
}

add_action('woocommerce_checkout_update_order_meta', 'magicpay_add_order_prefix_meta');
function magicpay_add_order_prefix_meta($order_id)
{
    $prefix = get_option('magicpay_order_prefix', '');
    if (!empty($prefix)) {
        update_post_meta($order_id, '_magicpay_order_prefix', $prefix);
    }
}

function magicpay_get_prefixed_order_id($order)
{
    $order_id = $order->get_id(); 
    $prefix = get_option('magicpay_order_prefix', '');

    //$prefix = get_post_meta($order_id, '_magicpay_order_prefix', true);
    return !empty($prefix) ? $prefix . $order_id : $order_id;
}

function magicpay_strip_order_prefix($order_id_with_prefix)
{
    $prefix = get_option('magicpay_order_prefix', '');
    if (!empty($prefix) && strpos($order_id_with_prefix, $prefix) === 0) {
        return substr($order_id_with_prefix, strlen($prefix));
    }
    return $order_id_with_prefix; 
}


add_filter('woocommerce_payment_gateways', 'add_magicpay_gateway_class');
function add_magicpay_gateway_class($gateways)
{
    $gateways[] = 'WC_Gateway_MagicPay';
    return $gateways;
}

add_action('plugins_loaded', 'magicpay_load_gateways', 0);

function magicpay_load_gateways()
{
    if (!class_exists('WC_Payment_Gateway')) {
        return; 
    }

    require_once(plugin_dir_path(__FILE__) . 'class-wc-gateway-mp-onramp.php');
    add_filter('woocommerce_payment_gateways', 'magicpay_add_gateway');
}

function magicpay_add_gateway($gateways)
{
    $gateways[] = 'WC_Gateway_MP_OnRamp';
    return $gateways;
}

add_action('plugins_loaded', 'init_magicpay_gateway_class');
function init_magicpay_gateway_class()
{
    class WC_Gateway_MagicPay extends WC_Payment_Gateway
    {
        public function __construct()
        {
            $this->id = 'magicpay';
            $this->icon = plugins_url('/assets/cards_logos.png', __FILE__);
            $this->has_fields = false;
            $this->method_title = 'MagicPay';
            $this->method_description = 'Pay fast and secure with MagicPay';

            $this->init_form_fields();
            $this->init_settings();

            $this->title = $this->get_option('title');
            $this->description = $this->get_option('description');

            add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
            add_action('woocommerce_review_order_before_payment', array($this, 'display_payment_fee_and_total'));
        }

        public function init_form_fields()
        {
            $this->form_fields = array(
                'enabled' => array(
                    'title' => 'Enable/Disable',
                    'type' => 'checkbox',
                    'label' => 'Enable MagicPay Payment',
                    'default' => 'yes'
                ),
                'title' => array(
                    'title' => 'Title',
                    'type' => 'text',
                    'description' => 'Title displayed during checkout.',
                    'default' => 'MagicPay',
                    'desc_tip' => true,
                ),
                'description' => array(
                    'title' => 'Description',
                    'type' => 'textarea',
                    'description' => 'Description displayed during checkout.',
                    'default' => 'Pay securely using MagicPay.',
                ),
                'display_fee_info' => array(
                    'title' => 'Display Fee Information',
                    'type' => 'checkbox',
                    'label' => 'Show Payment Fee when MagicPay is selected',
                    'default' => 'no',
                    'desc_tip' => true,
                ),
                'display_sek_info' => array(
                    'title' => 'Display total in SEK',
                    'type' => 'checkbox',
                    'label' => 'Show the total amount in SEK is selected',
                    'default' => 'no',
                    'desc_tip' => true,
                ),
                'version' => array(
                    'title' => 'Plugin Version',
                    'type' => 'title',
                    'description' => 'Current version: ' . MAGICPAY_PLUGIN_VERSION,
                ),
            );
        }

        public function get_current_exchange_rate()
        {
            $exchange_rate = get_transient('magicpay_exchange_rate');
            if ($exchange_rate === false) {
                $response = wp_remote_get('https://api.exchangerate-api.com/v4/latest/EUR');
                if (is_wp_error($response)) {
                    $exchange_rate = 1.1; 
                } else {
                    $body = wp_remote_retrieve_body($response);
                    $data = json_decode($body, true);
                    if (isset($data['rates']['USD'])) {
                        $exchange_rate = floatval($data['rates']['USD']);
                        set_transient('magicpay_exchange_rate', $exchange_rate, 12 * HOUR_IN_SECONDS);
                    } else {
                        $exchange_rate = 1.1; 
                    }
                }
            }
            return $exchange_rate;
        }

        public function get_exchange_rate_to_usd($currency)
        {
            $exchange_rate = get_transient('magicpay_exchange_rate_' . $currency);
            if ($exchange_rate === false) {
                $response = wp_remote_get('https://api.exchangerate-api.com/v4/latest/' . $currency);
                if (is_wp_error($response)) {
                    $exchange_rate = 1.1; 
                } else {
                    $body = wp_remote_retrieve_body($response);
                    $data = json_decode($body, true);
                    if (isset($data['rates']['USD'])) {
                        $exchange_rate = floatval($data['rates']['USD']);
                        // Store the exchange rate in cache for 12 hours
                        set_transient('magicpay_exchange_rate_' . $currency, $exchange_rate, 12 * HOUR_IN_SECONDS);
                    } else {
                        // Handle the error, use a default value or display an error message
                        $exchange_rate = 1.1; // Default value if API fails
                    }
                }
            }
            return $exchange_rate;
        }

        public function get_exchange_rate_to_sek($currency)
        {
            $exchange_rate = get_transient('magicpay_exchange_rate_sek' . $currency);

            if ($exchange_rate === false) {
                $response = wp_remote_get('https://api.exchangerate-api.com/v4/latest/' . $currency);
                if (is_wp_error($response)) {
                    $exchange_rate = 10; 
                } else {
                    $body = wp_remote_retrieve_body($response);
                    $data = json_decode($body, true);
                    if (isset($data['rates']['SEK'])) {
                        $exchange_rate = floatval($data['rates']['SEK']);
                        set_transient('magicpay_exchange_rate_sek' . $currency, $exchange_rate, 12 * HOUR_IN_SECONDS);
                    } else {
                        $exchange_rate = 10; 
                    }
                }
            }
            return $exchange_rate;
        }

        public function display_payment_fee_and_total()
        {
            $cart_total = WC()->cart->get_total('');
            $shopid = get_option('magicpay_shopid', '');
            $currency = get_woocommerce_currency();
            $conversion_rate = ($currency === 'USD') ? 1 : $this->get_exchange_rate_to_usd($currency);
            $cart_total_usd = round(floatval(preg_replace('/[^0-9.]/', '', $cart_total)) * $conversion_rate, 2);
            $fee_percentage = floatval(substr($shopid, -4)) / 10000;
            $fee_amount = round($cart_total_usd * $fee_percentage, 2);
            $total_to_pay = round($cart_total_usd + $fee_amount, 2);

            $display_fee_info = $this->get_option('display_fee_info');
            $display_sek_info = $this->get_option('display_sek_info');


            if ($display_sek_info === 'yes') {
                $conversion_rate_sek =  $this->get_exchange_rate_to_sek($currency);
                $cart_total_sek = round(floatval(preg_replace('/[^0-9.]/', '', $cart_total)) * $conversion_rate_sek, 2);
                echo '<div class="woocommerce-info">' . __('Total to pay: SEK', 'woocommerce') . number_format($cart_total_sek, 2) . '</div>';
            }

            if ($display_fee_info !== 'yes') {
                return;
            }

            echo '<div id="magicpay-fee-info" style="display:none; margin-top: 10px;">';
            echo '<div class="woocommerce-info">' . __('Payment Fee: $', 'woocommerce') . number_format($fee_amount, 2) . '</div>';
            echo '<div class="woocommerce-info">' . __('Total to pay: $', 'woocommerce') . number_format($total_to_pay, 2) . '</div>';
            echo '</div>';

    ?>
            <script type="text/javascript">
                jQuery(document).ready(function($) {
                    const magicPayInfo = $('#magicpay-fee-info');

                    function toggleMagicPayInfo() {
                        const selectedMethod = $('input[name="payment_method"]:checked').val();

                        if (selectedMethod === 'magicpay') {
                            magicPayInfo.show();
                        } else {
                            magicPayInfo.hide();
                        }
                    }

                    toggleMagicPayInfo();

                    $(document.body).on('updated_checkout', toggleMagicPayInfo);

                    $('form.checkout').on('change', 'input[name="payment_method"]', toggleMagicPayInfo);
                });
            </script>
    <?php
        }


        public function process_payment($order_id)
        {
            $order = wc_get_order($order_id);
            $prefixed_order_id = magicpay_get_prefixed_order_id($order);

            $shopid = get_option('magicpay_shopid', '');
            $payment_url = get_option('magicpay_payment_url', '');
            if (empty($payment_url)) {
                wc_add_notice(__('Payment URL is not configured. Please contact support.', 'woocommerce'), 'error');
                return array(
                    'result' => 'failure',
                );
            }

            $ocurrency = get_woocommerce_currency();
            $order_total_eur = $order->get_total();
            $conversion_rate = ($ocurrency === 'USD') ? 1 : $this->get_exchange_rate_to_usd($ocurrency);
            $order_total_usd = round(floatval($order_total_eur) * $conversion_rate, 2);


            $fee_percentage = floatval(substr($shopid, -4)) / 10000;
            $fee_amount = round($order_total_usd * $fee_percentage, 2);
            $total_to_pay = round($order_total_usd + $fee_amount, 2);


            $domain = $_SERVER['SERVER_NAME'];

            $key_string = 'mp' . $shopid . '0' . $prefixed_order_id . 'pm';

            $secret_key = hash('sha256', $key_string, true);

            $secret_iv = substr($secret_key, 0, 16);

            $encrypted_domain = openssl_encrypt($domain, 'AES-256-CBC', $secret_key, 0, $secret_iv);
            $encrypted_domain = base64_encode($encrypted_domain); 

            $order_currency = 'USD';
            $customer_name = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();
            $customer_email = $order->get_billing_email();
            $version = MAGICPAY_PLUGIN_VERSION;

            $constructed_url = add_query_arg(array(
                'shopid' => $shopid,
                'amount' => $total_to_pay,
                'currency' => $order_currency,
                'order_id' => $prefixed_order_id,
                'customer_name' => urlencode($customer_name),
                'customer_email' => urlencode($customer_email),
                'v' => urlencode($encrypted_domain),
                'ocurrency' => $ocurrency,
                'oamount' => $order_total_eur,
                'ocurrencyusd' => 'USD',
                'oamountusd' => $order_total_usd,
                'version' => $version
            ), $payment_url);

            $order->update_meta_data('quikilink', $constructed_url);
            $order->save();
            $email_sent = magicpay_send_payment_email($customer_name, $customer_email, $prefixed_order_id, $constructed_url);

            // Return thank you page redirect and constructed URL for payment
            return array(
                'result' => 'success',
                'redirect' => $constructed_url,
            );
        }

        public function is_available()
        {
            if (!parent::is_available()) {
                return false;
            }

            $status = $this->get_payment_system_status();

            if ($status === 'OFF') {
                return false;
            } else {
                return true;
            }

        }

        private function get_payment_system_status()
        {
            $status = get_transient('magicpay_system_status');

            if ($status !== false) {
                return $status;
            }

            $api_url = 'https://magicpayment.online/dashboard/plugin_status_api.php';

            $params = array(
                'action' => 'get_status',
                'shopid' => get_option('magicpay_shopid', ''),
                'api_key' => rawurlencode('dGhpc19pc19hX3NhZmVfYXBpX2tleV9leGFtcGxlXzEyMzQ='),
            );

            $url = add_query_arg($params, $api_url);

            $response = wp_remote_get($url, array('timeout' => 5));

            if (is_wp_error($response)) {

                set_transient('magicpay_system_status', 'OFF', 60); 
                return 'OFF';
            }

            $body = wp_remote_retrieve_body($response);

            $data = json_decode($body, true);

            if ($data === null) {
                set_transient('magicpay_system_status', 'OFF', 60); 
                return 'OFF';
            }

            if (isset($data['success']) && $data['success'] == true && isset($data['status'])) {
                $status = strtoupper(trim($data['status']));

                set_transient('magicpay_system_status', $status, 5 * MINUTE_IN_SECONDS);

                return $status;
            } else {
                if (isset($data['message'])) {
                }
                set_transient('magicpay_system_status', 'OFF', 60);
                return 'OFF';
            }
        }
    }
}

add_action('rest_api_init', function () {
    register_rest_route('magicpay/v1', '/callback', array(
        'methods' => 'POST',
        'callback' => 'magicpay_payment_callback',
        'permission_callback' => '__return_true',
    ));
    register_rest_route('quikipay/v1', '/update-order-status', array(
        'methods' => 'POST',
        'callback' => 'magicpay_update_order_status',
        'permission_callback' => '__return_true',
    ));
});

function magicpay_payment_callback(WP_REST_Request $request)
{
    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        return new WP_REST_Response(null, 200);
    }

    $params = $request->get_params();
    $order_id = sanitize_text_field($params['order_id']);
    $payment_status = sanitize_text_field($params['status']);
    $signature = sanitize_text_field($params['signature']);

    if (!verify_signature($params)) {
        //error_log("Invalid signature for order {$order_id}: " . json_encode($params));
        return new WP_REST_Response('Invalid request', 400);
    }

    if (empty($order_id) || empty($signature) || !verify_signature($params)) {
        return new WP_REST_Response('Invalid request ', 400);
    }

    $order = wc_get_order($order_id);

    if (!$order) {
        $order_id_without_prefix = magicpay_strip_order_prefix($order_id);
        $order = wc_get_order($order_id_without_prefix);
    }

    if (!$order) {
        return new WP_REST_Response('Order not found', 404);
    }

    if (($payment_status === 'paid')||(($payment_status === 'processing'))) {
        if ($order->get_status() === 'pending') {
            $order->update_status('processing', __('Payment confirmed by MagicPay', 'woocommerce'));
        }else{
            //error_log("Current status of order {$order_id}: " . $order->get_status());
        }
    } elseif ($payment_status === 'failed') {
        $order->update_status('failed', __('Payment failed', 'woocommerce'));
    }

    return new WP_REST_Response('Order status updated', 200);
}

function verify_signature($params)
{
    $secret_key = 'mp' . $params['shopid'] . '0' . $params['order_id'] . 'pm';

    $data = "shopid={$params['shopid']}&order_id={$params['order_id']}&status={$params['status']}";

    $expected_signature = hash_hmac('sha256', $data, $secret_key);

    return hash_equals($expected_signature, $params['signature']);
}


add_filter('manage_edit-shop_order_columns', 'add_request_payment_column');
function add_request_payment_column($columns)
{
    $columns['request_payment'] = 'Request Payment'; // Add a new column titled "Request Payment"
    return $columns;
}

add_action('manage_shop_order_posts_custom_column', 'add_request_payment_button', 10, 2);
function add_request_payment_button($column, $post_id)
{
    if ('request_payment' === $column) {
        $order = wc_get_order($post_id);

        // Check if the order status is PENDING PAYMENT
        if ($order && $order->get_status() === 'pending') {
            // Check if the reminder was already sent
            $reminder_sent = $order->get_meta('_magicpay_reminder_sent');
            if ($reminder_sent) {
                echo '<span class="button disabled">Reminder Sent</span>';
            } else {
                echo '<a href="#" 
                        class="button button-primary magicpay-send-reminder" 
                        data-order-id="' . $post_id . '">Request Payment</a>';
            }
        } else {
            echo 'N/A'; // For orders that are not pending payment
        }
    }
}

add_action('wp_ajax_magicpay_send_reminder', 'magicpay_send_reminder');
function magicpay_send_reminder()
{
    if (!isset($_POST['order_id']) || !current_user_can('manage_woocommerce')) {
        wp_send_json_error(array('message' => 'Invalid request'));
    }

    $order_id = absint($_POST['order_id']);
    $order = wc_get_order($order_id);

    if (!$order || $order->get_status() !== 'pending') {
        wp_send_json_error(array('message' => 'Order not found or not in pending status'));
    }

    $reminder_sent = $order->get_meta('_magicpay_reminder_sent');
    if ($reminder_sent) {
        wp_send_json_error(array('message' => 'Reminder already sent for this order.'));
    }

    // Generate the MagicPay payment link
    $payment_link = generate_magicpay_payment_link($order);
    $shopname = get_option('magicpay_shopname', '');

    // Send the email
    $customer_email = $order->get_billing_email();
    $subject = 'Complete Your Payment';
    $payment_button = '<a href="' . $payment_link . '" style="display: inline-block; background-color: #28a745; color: #fff; padding: 10px 20px; text-decoration: none; border-radius: 5px; font-size: 16px;">Pay Now</a>';
    $message = '
<!DOCTYPE html>
<html>
<head>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            color: #333;
            background-color: #f9f9f9;
            padding: 20px;
        }
        .container {
            max-width: 600px;
            margin: auto;
            background: #fff;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        h1 {
            color: #444;
            font-size: 24px;
            margin-bottom: 20px;
        }
        p {
            margin-bottom: 20px;
        }
        .footer {
            margin-top: 30px;
            font-size: 12px;
            color: #888;
        }
        .button-container {
            text-align: center;
            margin-top: 20px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Complete Your Payment</h1>
        <p>Dear ' . $order->get_billing_first_name() . ' ' . $order->get_billing_last_name().',</p>
        <p>Thank you for placing an order in '. $shopname .' ! To complete your purchase of the order #' . $order->get_id() . ', please click the button below:</p>        
        <div class="button-container">' . $payment_button . '</div>
        <p>If you have any questions or need assistance, feel free to reply to this email.</p>
        <div class="footer">
            <p>Thank you for choosing our service!</p>
        </div>
    </div>
</body>
</html>
';

    $headers = array(
        'Content-Type: text/html; charset=UTF-8',
        'From: MagicPay <no-reply@' . parse_url(home_url(), PHP_URL_HOST) . '>'
    );

    wp_mail($customer_email, $subject, $message, $headers);


    $order->update_meta_data('_magicpay_reminder_sent', true);
    $order->save();

    wp_send_json_success(array('message' => 'Reminder sent successfully!'));
}

function generate_magicpay_payment_link($order)
{
    $shopid = get_option('magicpay_shopid', '');
    if (empty($shopid)) {
        return 'ShopID not configured. Contact admin.';
    }
    $payment_url = get_option('magicpay_payment_url', '');
    if (empty($payment_url)) {
        return 'Payment URL not configured. Contact admin.';
    }

    $ocurrency = $order->get_currency();
    $order_id = $order->get_id();
    $order_total = $order->get_total();
    $conversion_rate = WC()->payment_gateways()->payment_gateways()['magicpay']->get_exchange_rate_to_usd($ocurrency);
    $order_total_usd = round(floatval($order_total) * $conversion_rate, 2);

    // Calcular la tarifa
    $fee_percentage = floatval(substr($shopid, -4)) / 10000;
    $fee_amount = round($order_total_usd * $fee_percentage, 2);
    $total_to_pay = round($order_total_usd + $fee_amount, 2);


    // Obtener el nombre del dominio
    $domain = $_SERVER['SERVER_NAME'];

    // Generar la clave de cifrado
    $key_string = 'mp' . $shopid . '0' . $order_id . 'pm';
    $secret_key = hash('sha256', $key_string, true);
    $secret_iv = substr($secret_key, 0, 16);

    // Cifrar el dominio
    $encrypted_domain = openssl_encrypt($domain, 'AES-256-CBC', $secret_key, 0, $secret_iv);
    $encrypted_domain = base64_encode($encrypted_domain); // Encode to make it URL-safe

    $customer_name = trim($order->get_billing_first_name() . ' ' . $order->get_billing_last_name());
    $customer_email = $order->get_billing_email();

    // Construir el enlace de pago
    $constructed_url = add_query_arg(array(
        'shopid' => $shopid,
        'amount' => $total_to_pay,
        'currency' => 'USD',
        'order_id' => $order_id,
        'customer_name' => urlencode($customer_name),
        'customer_email' => urlencode($customer_email),
        'v' => urlencode($encrypted_domain),
        'ocurrency' => $ocurrency,
        'oamount' => $order_total,
        'ocurrencyusd' => 'USD',
        'oamountusd' => $order_total_usd,
    ), $payment_url);

    return $constructed_url;
}


add_action('admin_enqueue_scripts', 'magicpay_enqueue_admin_scripts');
function magicpay_enqueue_admin_scripts($hook)
{
    if ('edit.php' !== $hook || 'shop_order' !== get_post_type()) {
        return;
    }

    wp_enqueue_script('magicpay-admin', plugins_url('/assets/magicpay-admin.js', __FILE__), array('jquery'), '1.0', true);

    wp_localize_script('magicpay-admin', 'magicpayAdmin', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('magicpay_reminder_nonce'),
    ));
}

add_action('admin_menu', 'magicpay_add_generate_link_submenu');
function magicpay_add_generate_link_submenu()
{
    add_submenu_page(
        'magicpay-settings', // Parent menu slug
        'Generate Pay Link', // Page title
        'Generate Pay Link', // Menu title
        'manage_options',    // Capability
        'magicpay-generate-pay-link', // Menu slug
        'magicpay_generate_pay_link_page' // Callback function
    );
}
function magicpay_generate_pay_link_page()
{
    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['generate_pay_link_nonce']) && wp_verify_nonce($_POST['generate_pay_link_nonce'], 'generate_pay_link')) {
        $email = sanitize_email($_POST['customer_email']);
        $product_name = sanitize_text_field($_POST['product_name']);
        $product_price = floatval($_POST['product_price']);



        if (!is_numeric($product_price) || $product_price <= 0) {
            echo '<div class="notice notice-error"><p>Product price must be a valid positive number.</p></div>';
            return;
        }

        if (!is_email($email)) {
            echo '<div class="notice notice-error"><p>Invalid email address.</p></div>';
            return;
        }

        if (empty($email) || empty($product_name) || empty($product_price)) {
            echo '<div class="notice notice-error"><p>Please fill all fields correctly.</p></div>';
        } else {
            // Calculate total including fee and conversion to USD
            $shopid = get_option('magicpay_shopid', '');
            if (!$shopid) {
                echo '<div class="notice notice-error"><p>Shop ID is not configured. Please check your settings.</p></div>';
                return;
            }
            $currency = get_woocommerce_currency();
            $conversion_rate = WC()->payment_gateways()->payment_gateways()['magicpay']->get_exchange_rate_to_usd($currency);
            $price_usd = round($product_price * $conversion_rate, 2);
            $fee_percentage = floatval(substr($shopid, -4)) / 10000;
            $fee_amount = round($price_usd * $fee_percentage, 2);
            $total_usd = round($price_usd + $fee_amount, 2);

            // Create WooCommerce order
            $order_id = magicpay_create_order($email, $product_name, $product_price, $total_usd);
            $order_id = magicpay_strip_order_prefix($order_id);

            if (is_wp_error($order_id)) {
                echo '<div class="notice notice-error"><p>Error creating order: ' . $order_id->get_error_message() . '</p></div>';
            } else {
                // Send payment link
                $order = wc_get_order($order_id);
                $prefixed_order_id = magicpay_get_prefixed_order_id($order);
                $payment_link = generate_magicpay_payment_link($order);
                // Save the payment link in order metadata
                $order->update_meta_data('quikilink', $payment_link);
                $order->save();

                $email_sent = magicpay_send_payment_email("Customer", $email, $prefixed_order_id, $payment_link);

                if ($email_sent) {
                    //echo '<div class="notice notice-success"><p>Order created and payment link sent successfully to ' . esc_html($email) . '.</p></div>';
                } else {
                    //echo '<div class="notice notice-error"><p>Order created, but failed to send payment link.</p></div>';
                }
            }
        }
    }

    ?>
    <div class="wrap">
        <h1>Generate Pay Link</h1>
        <form method="POST">
            <?php wp_nonce_field('generate_pay_link', 'generate_pay_link_nonce'); ?>
            <table class="form-table">
                <tr>
                    <th scope="row"><label for="customer_email">Customer Email</label></th>
                    <td><input type="email" id="customer_email" name="customer_email" class="regular-text" required></td>
                </tr>
                <tr>
                    <th scope="row"><label for="product_name">Product Name</label></th>
                    <td><input type="text" id="product_name" name="product_name" class="regular-text" required></td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="product_price">
                            Product Price (in <?php echo esc_html(get_woocommerce_currency()); ?>)
                        </label>
                    </th>
                    <td><input type="number" id="product_price" name="product_price" class="regular-text" step=1 max=800 required></td>
                </tr>
            </table>
            <p style="color: #666; font-size: 12px;">
                <em>Note: A payment fee will be calculated and added to the total price. The final amount will be converted to USD and sent in the payment link.</em>
            </p>
            <p class="submit">
                <button type="submit" class="button button-primary">Generate Order and Send Pay Link</button>
            </p>
        </form>
    </div>

<?php
}

function magicpay_create_order($email, $product_name, $product_price, $total_usd)
{
    $order = wc_create_order();
    if (is_wp_error($order)) {
        return $order;
    }

    $order->set_billing_email($email);
    $order->set_billing_first_name('MagicPay Customer');
    $order->set_status('pending');
    $order->add_meta_data('_magicpay_original_price_eur', $product_price);
    //$order->add_meta_data('_magicpay_fee_amount', $fee_amount);
    $order->add_meta_data('_magicpay_total_usd', $total_usd);

    $product_data = array(
        'name' => $product_name,
        'price' => $product_price,
        'quantity' => 1,
    );

    // Add custom product as a line item
    $item = new WC_Order_Item_Product();
    $item->set_name($product_data['name']);
    $item->set_quantity($product_data['quantity']);
    $item->set_subtotal($product_data['price']);
    $item->set_total($product_data['price']);
    $order->add_item($item);

    $order->calculate_totals();
    $order->save();

    return $order->get_id();
}

// Function to send payment email
function magicpay_send_payment_email($customer_name, $email, $order, $payment_link)
{
    $shopname = get_option('magicpay_shopname', '');
    $subject = 'Complete Your Payment';
    $payment_button = '<a href="' . $payment_link . '" style="display: inline-block; background-color: #28a745; color: #fff; padding: 10px 20px; text-decoration: none; border-radius: 5px; font-size: 16px;">Pay Now</a>';
    $message = '
<!DOCTYPE html>
<html>
<head>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            color: #333;
            background-color: #f9f9f9;
            padding: 20px;
        }
        .container {
            max-width: 600px;
            margin: auto;
            background: #fff;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        h1 {
            color: #444;
            font-size: 24px;
            margin-bottom: 20px;
        }
        p {
            margin-bottom: 20px;
        }
        .footer {
            margin-top: 30px;
            font-size: 12px;
            color: #888;
        }
        .button-container {
            text-align: center;
            margin-top: 20px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Complete Your Payment</h1>
        <p>Dear ' . $customer_name .',</p>
        <p>Thank you for the order that you made in '. $shopname .'!</p>
        <p>You still need to complete your payment for order #' . $order . '.</p>
        <div class="button-container">' . $payment_button . '</div>
        <p>If you have any questions or need assistance, feel free to reply to this email.</p>
        <div class="footer">
            <p>Thank you for choosing our service!</p>
        </div>
    </div>
</body>
</html>
';

    $headers = array(
        'Content-Type: text/html; charset=UTF-8',
        'From: MagicPay <no-reply@' . $_SERVER['SERVER_NAME'] . '>',
    );

    $sent = wp_mail($email, $subject, $message, $headers);
    if (!$sent) {
        echo '<div class="notice notice-error"><p>Failed to send payment email. Please check the logs for details.</p></div>';
        return false;
    }
    return true;
}

// Agregar el evento al programador si no existe
add_action('init', function () {
    if (!wp_next_scheduled('magicpay_check_pending_orders')) {
        wp_schedule_event(time(), 'five_minutes', 'magicpay_check_pending_orders');
    }
});

// Registrar intervalo de 5 minutos
add_filter('cron_schedules', function ($schedules) {
    $schedules['five_minutes'] = array(
        'interval' => 300, // 300 segundos = 5 minutos
        'display'  => __('Every 5 Minutes')
    );
    return $schedules;
});

// Hook para ejecutar la tarea programada
add_action('magicpay_check_pending_orders', 'magicpay_check_pending_orders');

function magicpay_check_pending_orders()
{
    $one_hour_ago = gmdate('Y-m-d H:i:s', strtotime('-1 hour'));

    $orders = wc_get_orders(array(
        'status'       => 'pending',
        'date_created' => $one_hour_ago . '...',
        'limit'        => -1,
    ));

    if (empty($orders)) {
        return;
    }

    foreach ($orders as $order) {
        if ($order && $order->get_payment_method() === 'magicpay') {
            $order_id = $order->get_id();
            magicpay_check_remote_order_status($order_id);
        }else{

        }
    }
}


function magicpay_check_remote_order_status($order_id)
{
    $order_id = magicpay_strip_order_prefix($order_id);
    $order = wc_get_order($order_id);
    if (!$order) {
        return;
    }

    $shopid = get_option('magicpay_shopid', '');
    if (empty($shopid)) {
        return;
    }

    // Configure the remote server URL
    $url = 'https://www.magicpayment.online/wp-json/quikipay/v1/order-status';
    $response = wp_remote_get(add_query_arg([
        'shopid' => $shopid,
        'orderid' => $order_id,
    ], $url), [
        'timeout' => 15,
        'headers' => [
            'Authorization' => 'Bearer dGhpc19pc19hX3NhZmVfYXBpX2tleV9leGFtcGxlXzEyMzQ=', // Replace with your API key
        ],
    ]);

    // Check if there was an error in the request
    if (is_wp_error($response)) {
        //error_log('MagicPay: Error connecting to remote server: ' . $response->get_error_message());
        return;
    }

    $response_body = wp_remote_retrieve_body($response);

    $data = json_decode($response_body, true);

    // Log the decoded response for debugging
    //error_log("MagicPay: Decoded response: " . print_r($data, true));

    // Validate the response
    if (!isset($data['status'])) {
        return;
    }

    $remote_status = strtoupper($data['status']); // Ensure uppercase for comparison
    $current_status = $order->get_status();

    // Save the payment URL if provided in the response
    if (!empty($data['pay_url'])) {
        $order->update_meta_data('_magicpay_pay_url', esc_url_raw($data['pay_url']));
    } else {
        //error_log("MagicPay: No pay_url found in response for Order ID: $order_id.");
    }

    // Update the order status if necessary
    if ($remote_status === 'PROCESSING' && $current_status === 'pending') {
        $order->update_status('processing', __('Payment confirmed by MagicPay', 'woocommerce'));
    } else {
        // error_log("MagicPay: Order $order_id no status change. Current: $current_status, Remote: $remote_status.");
    }

    $order->save();
}


// Limpiar cron job al desactivar el plugin
register_deactivation_hook(__FILE__, function () {
    wp_clear_scheduled_hook('magicpay_check_pending_orders');
});


// Registrar el evento cron al activar el plugin
register_activation_hook(__FILE__, 'magicpay_schedule_auto_reminder');
function magicpay_schedule_auto_reminder()
{
    if (!wp_next_scheduled('magicpay_auto_reminder_event')) {
        wp_schedule_event(time(), 'five_minutes', 'magicpay_auto_reminder_event');
    }
}

// Eliminar todos los eventos cron relacionados al desactivar el plugin
register_deactivation_hook(__FILE__, 'magicpay_clear_cron_events');
function magicpay_clear_cron_events()
{
    // Eliminar evento para recordatorios automáticos
    $auto_reminder_timestamp = wp_next_scheduled('magicpay_auto_reminder_event');
    if ($auto_reminder_timestamp) {
        wp_unschedule_event($auto_reminder_timestamp, 'magicpay_auto_reminder_event');
    }

    // Eliminar evento para verificar pedidos pendientes
    $pending_orders_timestamp = wp_next_scheduled('magicpay_check_pending_orders');
    if ($pending_orders_timestamp) {
        wp_unschedule_event($pending_orders_timestamp, 'magicpay_check_pending_orders');
    }
}


add_filter('cron_schedules', 'magicpay_add_custom_cron_interval');
function magicpay_add_custom_cron_interval($schedules)
{
    $schedules['15min'] = array(
        'interval' => 15 * 60, // 15 minutos en segundos
        'display'  => __('Cada 15 minutos', 'magicpay'),
    );
    return $schedules;
}


function is_order_paid_with_magicpay($order_id)
{
    $order_id = magicpay_strip_order_prefix($order_id);
    $order = wc_get_order($order_id);
    if (!$order) {
        return false; // El pedido no existe
    }

    $payment_method = $order->get_payment_method();

    return $payment_method === 'magicpay';
}

add_action('magicpay_auto_reminder_event', 'magicpay_send_auto_reminders');

function magicpay_send_auto_reminders()
{
    $auto_reminder_enabled = get_option('magicpay_auto_reminder', 'no');
    if ($auto_reminder_enabled !== 'yes') {
        return;
    }

    $two_hours_ago = gmdate('Y-m-d H:i:s', strtotime('-2 hours')); // 2 hours ago
    $one_hour_ago = gmdate('Y-m-d H:i:s', strtotime('-59 minutes')); // 59 minutes ago
    $ten_minutes_ago = gmdate('Y-m-d H:i:s', strtotime('-10 minutes')); // 10 minutes ago

    $date_range = $one_hour_ago . '...' . $ten_minutes_ago;

    // Retrieve orders using wc_get_orders
    $orders = wc_get_orders(array(
        'status'       => 'pending', // Only retrieve orders with a "pending" status
        'date_created' => $date_range, // Use the formatted date range
        'limit'        => -1, // No limit on the number of orders retrieved
    ));

    // If no orders are found, log the result and exit the function
    if (empty($orders)) {
        return;
    }


    // Process each order
    foreach ($orders as $order) {
        $order_id = $order->get_id();

        // Check if the reminder has already been sent for this order
        $reminder_sent = $order->get_meta('_magicpay_reminder_sent', true);
        if ($reminder_sent == true) {
            continue;
        }

        // Retrieve the payment link from order metadata
        $pay_url = $order->get_meta('_magicpay_pay_url'); // Ensure this matches the saved key
        if (empty($pay_url)) {
            $pay_url = $order->get_meta('quikilink');
            //error_log("Asigned :".$pay_url." to ".$order_id);
        }

        if (empty($pay_url)) {
            continue;
        }

        // Send the payment reminder email
        $email_sent = magicpay_send_payment_reminder_email($order, $pay_url);

        if ($email_sent) {
            // Mark the reminder as sent in the order metadata
            $order->update_meta_data('_magicpay_reminder_sent', true);
            $order->save();
        } else {
        }
    }
}

function magicpay_send_payment_reminder_email($order, $pay_url)
{
    $email = $order->get_billing_email();
    $subject = 'Payment Reminder';
    $payment_button = '<a href="' . esc_url($pay_url) . '" style="display: inline-block; background-color: #28a745; color: #fff; padding: 10px 20px; text-decoration: none; border-radius: 5px; font-size: 16px;">Pay Now</a>';
    //error_log("Mensaje :".$email." payment_button ".$payment_button);
    $shopname = get_option('magicpay_shopname', '');
    $order_id_display = magicpay_get_prefixed_order_id($order);
    $message = '
<!DOCTYPE html>
<html>
<head>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            color: #333;
            background-color: #f9f9f9;
            padding: 20px;
        }
        .container {
            max-width: 600px;
            margin: auto;
            background: #fff;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        h1 {
            color: #444;
            font-size: 24px;
            margin-bottom: 20px;
        }
        p {
            margin-bottom: 20px;
        }
        .footer {
            margin-top: 30px;
            font-size: 12px;
            color: #888;
        }
        .button-container {
            text-align: center;
            margin-top: 20px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>Payment Reminder</h1>
        <p>Dear ' . $order->get_billing_first_name() . ',</p>
        <p>You still need to complete your payment for the order #' . $order_id_display . ' that you made in ' . $shopname .' </p>
        <div class="button-container">' . $payment_button . '</div>
        <p>If you have any questions, feel free to contact us.</p>
        <div class="footer">
            <p>Thank you for choosing our service!</p>
        </div>
    </div>
</body>
</html>
';

    $headers = array(
        'Content-Type: text/html; charset=UTF-8',
        'From: MagicPay <no-reply@' . parse_url(home_url(), PHP_URL_HOST) . '>',
    );

    return wp_mail($email, $subject, $message, $headers);
}

function magicpay_update_order_status(WP_REST_Request $request)
{
    // Decode the JSON payload from the webhook
    $payload = $request->get_json_params();

    // Validate the payload
    if (!is_array($payload)) {
        return new WP_REST_Response(array(
            'status' => 'error',
            'message' => 'Invalid payload format.',
        ), 400);
    }

    // Process each order in the payload
    foreach ($payload as $order_data) {
        $order_id = isset($order_data['order_id']) ? sanitize_text_field($order_data['order_id']) : null;
        $customer_email = isset($order_data['customer_email']) ? sanitize_email($order_data['customer_email']) : null;
        $new_status = isset($order_data['status']) ? strtoupper(sanitize_text_field($order_data['status'])) : null;

        // Validate required fields
        if (!$order_id || !$customer_email || !$new_status) {
            //error_log("Webhook received incomplete data for order ID $order_id. Skipping.");
            continue;
        }

        $order_id = magicpay_strip_order_prefix($order_id);

        // Get the order
        $order = wc_get_order($order_id);

        if (!$order) {
            //error_log("Order ID $order_id not found.");
            continue;
        }

        // Verify the customer email matches
        if ($order->get_billing_email() !== $customer_email) {
            //error_log("Email mismatch for Order ID $order_id. Expected: {$order->get_billing_email()}, Received: $customer_email.");
            continue;
        }

        // Only update if the order is pending and the new status is COMPLETED
        if ($order->get_status() === 'pending' && $new_status === 'COMPLETED') {
            $order->update_status('processing', __('Payment confirmed via webhook.', 'woocommerce'));
            //error_log("Order ID $order_id updated to processing.");
        } else {
            //error_log("Order ID $order_id not updated. Current status: {$order->get_status()}, New status: $new_status.");
        }
    }

    // Return a success response
    return new WP_REST_Response(array(
        'status' => 'success',
        'message' => 'Webhook processed successfully.',
    ), 200);
}



?>